﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.EntityModels;
   using Hims.Shared.Library.Enums;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Shared.DataFilters;
    using Shared.UserModels.Pharmacy;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/pharmacy-department")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class PharmacyDepartmentController : BaseController
    {
        /// <summary>
        /// The pharmacy department service.
        /// </summary>
        private readonly IPharmacyDepartmentService pharmacyDepartmentService;
        /// <summary>
        /// The pharmacy service.
        /// </summary>
        private readonly IPharmacyLogService pharmacyLogService;

        /// <inheritdoc />
        public PharmacyDepartmentController(IPharmacyDepartmentService pharmacyDepartmentService, IPharmacyLogService pharmacyLogService)
        {
            this.pharmacyDepartmentService = pharmacyDepartmentService;
            this.pharmacyLogService = pharmacyLogService;
        }

        /// <summary>
        /// Modifies the pharmacy department asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify")]
        public async Task<ActionResult> ModifyPharmacyDepartmentAsync([FromBody] PharmacyDepartmentModel model, [FromHeader] LocationHeader header)
        {
            model = (PharmacyDepartmentModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;           
            var response = await this.pharmacyDepartmentService.ModifyDepartment(model);
            if (response > 0)
            {
                var pharmacyLogModel = new PharmacyLogModel
                {
                    AccountId = model.LoginAccountId,
                    PharmacyLogTypeId = (int)PharmacyLogTypes.Department,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@"<b>{model.ModifiedByName}</b>  {(model.PharmacyDepartmentId > 0 ? "updated" : "added")} the Department Name - <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("MM/dd/yyyy, hh:mm tt")} successfully",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.pharmacyLogService.LogAsync(pharmacyLogModel);
            }
            
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the pharmacy department asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-pharmacy-department")]
        public async Task<ActionResult> FetchPharmacyDepartmentAsync([FromBody] PharmacyDepartmentModel model, [FromHeader] LocationHeader header)
        {
            model ??= new PharmacyDepartmentModel();
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var response = await this.pharmacyDepartmentService.FetchPharmacyDepartmentAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the department stocks.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-department-stock")]
        public async Task<ActionResult> FetchDepartmentStocks([FromBody] PharmacyDepartmentalStockModel model, [FromHeader] LocationHeader header)
        {
            model ??= new PharmacyDepartmentalStockModel();
            model.LocationId = !string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null;
            var response = await this.pharmacyDepartmentService.FetchDepartmentalStocks(model);
            return this.Success(response);
        }

        /// <summary>
        /// Modifies the product details asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify-product-detail")]
        public async Task<ActionResult> ModifyProductDetailsAsync([FromBody] PharmacyProductDetailModel model, [FromHeader] LocationHeader header)
        {
            model = (PharmacyProductDetailModel)EmptyFilter.Handler(model);
            var response = await this.pharmacyDepartmentService.AddPharmacyProductAdditionalDetailsAsync(model);
            if (response > 0)
            {
                var pharmacyLogModel = new PharmacyLogModel
                {
                    AccountId = model.LoginAccountId,
                    PharmacyLogTypeId = (int)(model.RetailPharmacyId== null? PharmacyLogTypes.Central_Store_Stock : PharmacyLogTypes.Retail_Store_Stock),
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@"<b>{model.ModifiedByName}</b> has Updated ProductDetails - <b>{model.ProductName}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("MM/dd/yyyy, hh:mm tt")} Successfully",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.pharmacyLogService.LogAsync(pharmacyLogModel);
            }
                return this.Success(response);
        }
        
        /// <summary>
        /// Adds the departmental consumption asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("add-departmental-consumption")]
        public async Task<ActionResult> AddDepartmentalConsumptionAsync([FromBody] DepartmentConsumptionModel model)
        {
            model = (DepartmentConsumptionModel)EmptyFilter.Handler(model);
            var response = await this.pharmacyDepartmentService.AddDepartmentStockConsumptionAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the departmental consumption asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-departmental-consumption")]
        public async Task<ActionResult> FetchDepartmentalConsumptionAsync([FromBody] DepartmentConsumptionModel model)
        {
            model = (DepartmentConsumptionModel)EmptyFilter.Handler(model);
            var response = await this.pharmacyDepartmentService.FetchDepartmentalConsumptionsAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the departmental consumption history asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-departmental-consumption-history")]
        public async Task<ActionResult> FetchDepartmentalConsumptionHistoryAsync([FromBody] DepartmentConsumptionModel model)
        {
            model = (DepartmentConsumptionModel)EmptyFilter.Handler(model);
            var response = await this.pharmacyDepartmentService.FetchDepartmentalConsumptionProductsAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches the pharmacy racks asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("get-pharmacy-rack")]
        public async Task<ActionResult> FetchPharmacyRacksAsync([FromBody] PharmacyProductRackModel model)
        {
            model = (PharmacyProductRackModel)EmptyFilter.Handler(model);
            var response = await this.pharmacyDepartmentService.FetchRackDepartmentWiseAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// modify the department status asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify-department-status")]
        public async Task<ActionResult> ModifyDepartmentStatusAsync([FromBody] PharmacyDepartmentModel model, [FromHeader] LocationHeader header)
        {
            model =(PharmacyDepartmentModel)EmptyFilter.Handler(model);
            var response = await this.pharmacyDepartmentService.ActivateOrDeactivateDepartment(model);
            if (response == 0)
            {
                return this.ServerError();
            }
            if (response > 0)
            {
                var pharmacyLogModel = new PharmacyLogModel
                {
                    AccountId = model.LoginAccountId,
                    PharmacyLogTypeId = (int)PharmacyLogTypes.Department,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@"<b>{model.ModifiedByName}</b> has {((bool)model.Active ? "Activated" : "Deactivated")} the <b>{model.Name}  </b> on {DateTime.UtcNow.AddMinutes(330).ToString("MM/dd/yyyy, hh:mm tt")} successfully",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.pharmacyLogService.LogAsync(pharmacyLogModel);
            }

            return this.Success(response);
        }
    }
}